<?php

require_once __DIR__ . '/../config/database.php';

/**
 * DatabaseMigration Class
 * Veritabanı migration işlemlerini yöneten sınıf
 */
class DatabaseMigration {
    private $database;
    private $backupPath;
    private $logFile;
    
    public function __construct($database = null) {
        $this->database = $database ?? Database::getInstance();
        $this->backupPath = __DIR__ . '/../backups/';
        $this->logFile = __DIR__ . '/../logs/migration.log';
        
        // Backup ve log dizinlerini oluştur
        if (!file_exists($this->backupPath)) {
            mkdir($this->backupPath, 0755, true);
        }
        
        $logDir = dirname($this->logFile);
        if (!file_exists($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }
    
    /**
     * Veritabanı yedeği oluştur
     * @return array İşlem sonucu
     */
    public function createBackup(): array {
        try {
            $timestamp = date('Y-m-d_H-i-s');
            $backupFile = $this->backupPath . "backup_{$timestamp}.sql";
            
            $this->log("Backup başlatılıyor: $backupFile");
            
            // Veritabanı bilgilerini al
            $host = 'localhost';
            $dbName = 'depo_stok_takip';
            $username = 'root';
            $password = '';
            
            // mysqldump komutu
            $command = sprintf(
                'mysqldump --host=%s --user=%s --password=%s --single-transaction --routines --triggers %s > %s 2>&1',
                escapeshellarg($host),
                escapeshellarg($username),
                $password ? escapeshellarg($password) : "''",
                escapeshellarg($dbName),
                escapeshellarg($backupFile)
            );
            
            // Windows için mysqldump yolu
            $mysqldumpPath = 'C:\\xampp4\\mysql\\bin\\mysqldump.exe';
            if (file_exists($mysqldumpPath)) {
                $command = sprintf(
                    '"%s" --host=%s --user=%s %s --single-transaction --routines --triggers %s > %s 2>&1',
                    $mysqldumpPath,
                    escapeshellarg($host),
                    escapeshellarg($username),
                    $password ? '--password=' . escapeshellarg($password) : '',
                    escapeshellarg($dbName),
                    escapeshellarg($backupFile)
                );
            }
            
            exec($command, $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new Exception("Backup komutu başarısız: " . implode("\n", $output));
            }
            
            if (!file_exists($backupFile) || filesize($backupFile) === 0) {
                throw new Exception("Backup dosyası oluşturulamadı veya boş");
            }
            
            $fileSize = filesize($backupFile);
            $this->log("Backup başarıyla oluşturuldu: $backupFile (Boyut: " . $this->formatBytes($fileSize) . ")");
            
            return [
                'success' => true,
                'backup_file' => $backupFile,
                'file_size' => $fileSize,
                'timestamp' => $timestamp,
                'message' => 'Veritabanı yedeği başarıyla oluşturuldu'
            ];
            
        } catch (Exception $e) {
            $this->log("Backup hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Backup oluşturulurken hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Backup'tan veritabanını geri yükle
     * @param string $backupFile Backup dosya yolu
     * @return array İşlem sonucu
     */
    public function restoreBackup(string $backupFile): array {
        try {
            if (!file_exists($backupFile)) {
                throw new Exception("Backup dosyası bulunamadı: $backupFile");
            }
            
            $this->log("Backup restore başlatılıyor: $backupFile");
            
            // Veritabanı bilgilerini al
            $host = 'localhost';
            $dbName = 'depo_stok_takip';
            $username = 'root';
            $password = '';
            
            // mysql komutu
            $command = sprintf(
                'mysql --host=%s --user=%s --password=%s %s < %s 2>&1',
                escapeshellarg($host),
                escapeshellarg($username),
                $password ? escapeshellarg($password) : "''",
                escapeshellarg($dbName),
                escapeshellarg($backupFile)
            );
            
            // Windows için mysql yolu
            $mysqlPath = 'C:\\xampp4\\mysql\\bin\\mysql.exe';
            if (file_exists($mysqlPath)) {
                $command = sprintf(
                    '"%s" --host=%s --user=%s %s %s < %s 2>&1',
                    $mysqlPath,
                    escapeshellarg($host),
                    escapeshellarg($username),
                    $password ? '--password=' . escapeshellarg($password) : '',
                    escapeshellarg($dbName),
                    escapeshellarg($backupFile)
                );
            }
            
            exec($command, $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new Exception("Restore komutu başarısız: " . implode("\n", $output));
            }
            
            $this->log("Backup başarıyla restore edildi: $backupFile");
            
            return [
                'success' => true,
                'backup_file' => $backupFile,
                'message' => 'Veritabanı başarıyla geri yüklendi'
            ];
            
        } catch (Exception $e) {
            $this->log("Restore hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Restore işlemi sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Veri tipi migration'ı gerçekleştir
     * @return array İşlem sonucu
     */
    public function migrateDataTypes(): array {
        try {
            $this->log("Veri tipi migration başlatılıyor");
            
            $this->database->beginTransaction();
            
            $migrations = [
                // cariler tablosu
                "ALTER TABLE cariler MODIFY COLUMN cari_bakiye DECIMAL(15,2) DEFAULT 0.00",
                "ALTER TABLE cariler MODIFY COLUMN bakiye DECIMAL(15,2) DEFAULT 0.00",
                
                // cari_hareketler tablosu
                "ALTER TABLE cari_hareketler MODIFY COLUMN tutar DECIMAL(15,2) NOT NULL",
                
                // satislar tablosu
                "ALTER TABLE satislar MODIFY COLUMN toplam_tutar DECIMAL(15,2) NOT NULL",
                "ALTER TABLE satislar MODIFY COLUMN odenen_tutar DECIMAL(15,2) DEFAULT 0.00",
                "ALTER TABLE satislar MODIFY COLUMN kalan_tutar DECIMAL(15,2) DEFAULT 0.00",
                
                // stok_hareketleri tablosu
                "ALTER TABLE stok_hareketleri MODIFY COLUMN birim_fiyat DECIMAL(15,2) DEFAULT NULL",
                "ALTER TABLE stok_hareketleri MODIFY COLUMN toplam_tutar DECIMAL(15,2) DEFAULT NULL",
                
                // urunler tablosu
                "ALTER TABLE urunler MODIFY COLUMN alis_fiyati DECIMAL(15,2) NOT NULL",
                "ALTER TABLE urunler MODIFY COLUMN satis_fiyati DECIMAL(15,2) NOT NULL"
            ];
            
            $successCount = 0;
            foreach ($migrations as $sql) {
                try {
                    $this->database->execute($sql);
                    $successCount++;
                    $this->log("Migration başarılı: " . substr($sql, 0, 50) . "...");
                } catch (Exception $e) {
                    $this->log("Migration hatası: $sql - " . $e->getMessage(), 'ERROR');
                    throw $e;
                }
            }
            
            $this->database->commit();
            
            $this->log("Veri tipi migration tamamlandı: $successCount migration başarılı");
            
            return [
                'success' => true,
                'migrations_applied' => $successCount,
                'total_migrations' => count($migrations),
                'message' => 'Veri tipi migration başarıyla tamamlandı'
            ];
            
        } catch (Exception $e) {
            $this->database->rollback();
            $this->log("Migration hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Migration sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Trigger'ları güncelle
     * @return array İşlem sonucu
     */
    public function updateTriggers(): array {
        try {
            $this->log("Trigger güncelleme başlatılıyor");
            
            // Önce mevcut trigger'ları sil
            $dropTriggers = [
                "DROP TRIGGER IF EXISTS satis_sonrasi_cari_hareket",
                "DROP TRIGGER IF EXISTS satis_silme_sonrasi_cari_hareket",
                "DROP TRIGGER IF EXISTS update_stock_after_movement"
            ];
            
            foreach ($dropTriggers as $sql) {
                $this->database->execute($sql);
                $this->log("Trigger silindi: $sql");
            }
            
            // Yeni trigger'ları oluştur
            $createTriggers = [
                // satis_sonrasi_cari_hareket - Güncellenmiş versiyon
                "CREATE TRIGGER satis_sonrasi_cari_hareket AFTER INSERT ON satislar FOR EACH ROW
                BEGIN
                    -- Borç kaydı (toplam tutar)
                    INSERT INTO cari_hareketler (cari_id, hareket_tipi, tutar, aciklama, satis_id, tarih)
                    VALUES (NEW.cari_id, 'borc', NEW.toplam_tutar, CONCAT('Satış No: ', NEW.satis_id), NEW.satis_id, NEW.satis_tarihi);
                    
                    -- Alacak kaydı (ödenen tutar, eğer varsa)
                    IF NEW.odenen_tutar > 0 THEN
                        INSERT INTO cari_hareketler (cari_id, hareket_tipi, tutar, aciklama, satis_id, tarih)
                        VALUES (NEW.cari_id, 'alacak', NEW.odenen_tutar, CONCAT('Satış No: ', NEW.satis_id, ' - Ödeme'), NEW.satis_id, NEW.satis_tarihi);
                    END IF;
                    
                    -- Cari bakiyesini güncelle (toplam tutardan ödenen tutarı çıkararak)
                    UPDATE cariler 
                    SET cari_bakiye = cari_bakiye + (NEW.toplam_tutar - NEW.odenen_tutar),
                        bakiye = bakiye + (NEW.toplam_tutar - NEW.odenen_tutar)
                    WHERE cari_id = NEW.cari_id;
                END",
                
                // satis_silme_sonrasi_cari_hareket
                "CREATE TRIGGER satis_silme_sonrasi_cari_hareket AFTER DELETE ON satislar FOR EACH ROW
                BEGIN
                    DELETE FROM cari_hareketler WHERE satis_id = OLD.satis_id;
                    
                    UPDATE cariler 
                    SET cari_bakiye = cari_bakiye - OLD.toplam_tutar,
                        bakiye = bakiye - OLD.toplam_tutar
                    WHERE cari_id = OLD.cari_id;
                END",
                
                // update_stock_after_movement
                "CREATE TRIGGER update_stock_after_movement AFTER INSERT ON stok_hareketleri FOR EACH ROW
                BEGIN
                    UPDATE urunler 
                    SET stok_miktari = NEW.yeni_stok 
                    WHERE urun_id = NEW.urun_id;
                END"
            ];
            
            $successCount = 0;
            foreach ($createTriggers as $sql) {
                try {
                    $this->database->execute($sql);
                    $successCount++;
                    $this->log("Trigger oluşturuldu");
                } catch (Exception $e) {
                    $this->log("Trigger oluşturma hatası: " . $e->getMessage(), 'ERROR');
                    throw $e;
                }
            }
            
            $this->log("Trigger güncelleme tamamlandı: $successCount trigger güncellendi");
            
            return [
                'success' => true,
                'triggers_updated' => $successCount,
                'message' => 'Trigger\'lar başarıyla güncellendi'
            ];
            
        } catch (Exception $e) {
            $this->log("Trigger güncelleme hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Trigger güncelleme sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * View'ları güncelle
     * @return array İşlem sonucu
     */
    public function updateViews(): array {
        try {
            $this->log("View güncelleme başlatılıyor");
            
            // Önce mevcut view'ları sil
            $dropViews = [
                "DROP VIEW IF EXISTS cari_bakiye_ozeti",
                "DROP VIEW IF EXISTS hizli_cari_bakiye",
                "DROP VIEW IF EXISTS stok_durumu_ozet",
                "DROP VIEW IF EXISTS hizli_stok_durumu"
            ];
            
            foreach ($dropViews as $sql) {
                $this->database->execute($sql);
                $this->log("View silindi: $sql");
            }
            
            // Yeni view'ları oluştur (güncellenmiş veri tipleriyle)
            $createViews = [
                // cari_bakiye_ozeti
                "CREATE VIEW cari_bakiye_ozeti AS
                SELECT 
                    c.cari_id,
                    c.firma_adi,
                    c.cari_tipi,
                    COALESCE(SUM(CASE WHEN ch.hareket_tipi = 'borc' THEN ch.tutar ELSE 0 END), 0) AS toplam_borc,
                    COALESCE(SUM(CASE WHEN ch.hareket_tipi = 'alacak' THEN ch.tutar ELSE 0 END), 0) AS toplam_alacak,
                    COALESCE(SUM(CASE WHEN ch.hareket_tipi = 'borc' THEN ch.tutar ELSE -ch.tutar END), 0) AS bakiye
                FROM cariler c
                LEFT JOIN cari_hareketler ch ON c.cari_id = ch.cari_id
                GROUP BY c.cari_id, c.firma_adi, c.cari_tipi",
                
                // hizli_cari_bakiye
                "CREATE VIEW hizli_cari_bakiye AS
                SELECT 
                    c.cari_id,
                    c.firma_adi,
                    c.cari_tipi,
                    c.cari_bakiye,
                    CASE 
                        WHEN c.cari_bakiye > 0 THEN 'Borçlu'
                        WHEN c.cari_bakiye < 0 THEN 'Alacaklı'
                        ELSE 'Sıfır'
                    END AS durum
                FROM cariler c",
                
                // stok_durumu_ozet
                "CREATE VIEW stok_durumu_ozet AS
                SELECT 
                    u.urun_id,
                    u.urun_adi,
                    u.barkod,
                    u.stok_miktari,
                    u.alis_fiyati,
                    u.satis_fiyati,
                    COALESCE(sus.minimum_stok, 0) AS minimum_stok,
                    COALESCE(sus.maksimum_stok, 0) AS maksimum_stok,
                    COALESCE(sus.kritik_seviye, 0) AS kritik_seviye,
                    CASE 
                        WHEN u.stok_miktari <= COALESCE(sus.kritik_seviye, 0) THEN 'Kritik'
                        WHEN u.stok_miktari <= COALESCE(sus.minimum_stok, 0) THEN 'Düşük'
                        WHEN u.stok_miktari >= COALESCE(sus.maksimum_stok, 999999) THEN 'Yüksek'
                        ELSE 'Normal'
                    END AS stok_durumu,
                    u.stok_miktari * u.alis_fiyati AS toplam_alis_degeri,
                    u.stok_miktari * u.satis_fiyati AS toplam_satis_degeri
                FROM urunler u
                LEFT JOIN stok_uyari_seviyeleri sus ON u.urun_id = sus.urun_id",
                
                // hizli_stok_durumu
                "CREATE VIEW hizli_stok_durumu AS
                SELECT 
                    u.urun_id,
                    u.urun_adi,
                    u.stok_miktari,
                    u.alis_fiyati,
                    u.satis_fiyati,
                    u.satis_fiyati - u.alis_fiyati AS kar_marji,
                    CASE 
                        WHEN u.stok_miktari <= 5 THEN 'Kritik'
                        WHEN u.stok_miktari <= 20 THEN 'Düşük'
                        WHEN u.stok_miktari <= 100 THEN 'Normal'
                        ELSE 'Yüksek'
                    END AS stok_durumu,
                    u.stok_miktari * u.alis_fiyati AS toplam_deger
                FROM urunler u"
            ];
            
            $successCount = 0;
            foreach ($createViews as $sql) {
                try {
                    $this->database->execute($sql);
                    $successCount++;
                    $this->log("View oluşturuldu");
                } catch (Exception $e) {
                    $this->log("View oluşturma hatası: " . $e->getMessage(), 'ERROR');
                    throw $e;
                }
            }
            
            $this->log("View güncelleme tamamlandı: $successCount view güncellendi");
            
            return [
                'success' => true,
                'views_updated' => $successCount,
                'message' => 'View\'lar başarıyla güncellendi'
            ];
            
        } catch (Exception $e) {
            $this->log("View güncelleme hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'View güncelleme sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Migration'ı geri al
     * @param string $backupFile Geri yüklenecek backup dosyası
     * @return array İşlem sonucu
     */
    public function rollback(string $backupFile): array {
        $this->log("Rollback başlatılıyor: $backupFile");
        return $this->restoreBackup($backupFile);
    }
    
    /**
     * Migration durumunu kontrol et
     * @return array Doğrulama sonucu
     */
    public function validateMigration(): array {
        try {
            $this->log("Migration validation başlatılıyor");
            
            $validations = [];
            
            // Veri tipi kontrolü
            $tables = ['cariler', 'cari_hareketler', 'satislar', 'stok_hareketleri', 'urunler'];
            foreach ($tables as $table) {
                $sql = "SHOW COLUMNS FROM $table";
                $columns = $this->database->fetchAll($sql);
                
                foreach ($columns as $column) {
                    if (strpos($column['Type'], 'decimal') !== false) {
                        $validations[] = [
                            'table' => $table,
                            'column' => $column['Field'],
                            'type' => $column['Type'],
                            'valid' => strpos($column['Type'], 'decimal(15,2)') !== false
                        ];
                    }
                }
            }
            
            // Trigger kontrolü
            $sql = "SHOW TRIGGERS";
            $triggers = $this->database->fetchAll($sql);
            $validations['triggers'] = count($triggers);
            
            // View kontrolü
            $sql = "SHOW FULL TABLES WHERE Table_type = 'VIEW'";
            $views = $this->database->fetchAll($sql);
            $validations['views'] = count($views);
            
            $this->log("Migration validation tamamlandı");
            
            return [
                'success' => true,
                'validations' => $validations,
                'message' => 'Migration validation başarılı'
            ];
            
        } catch (Exception $e) {
            $this->log("Validation hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Validation sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Log mesajı yaz
     * @param string $message Mesaj
     * @param string $level Log seviyesi
     */
    private function log(string $message, string $level = 'INFO'): void {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] [$level] $message\n";
        file_put_contents($this->logFile, $logMessage, FILE_APPEND);
    }
    
    /**
     * Byte'ları okunabilir formata çevir
     * @param int $bytes Byte sayısı
     * @return string Formatlanmış boyut
     */
    private function formatBytes(int $bytes): string {
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = 0;
        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }
        return round($bytes, 2) . ' ' . $units[$i];
    }
}

?>
