<?php

require_once __DIR__ . '/../config/database.php';

if (session_status() == PHP_SESSION_NONE) {
    ini_set('session.cookie_httponly', 1);
    ini_set('session.cookie_secure', 1);
    ini_set('session.use_only_cookies', 1);
    ini_set('session.cookie_samesite', 'Strict');
    session_start();
}

class Security {
    
    public static function generateCSRFToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    public static function validateCSRFToken($token) {
        if (!isset($_SESSION['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $token)) {
            return false;
        }
        return true;
    }

    public static function getCSRFTokenInput() {
        $token = self::generateCSRFToken();
        return '<input type="hidden" name="csrf_token" value="' . htmlspecialchars($token, ENT_QUOTES, 'UTF-8') . '">';
    }

    public static function sanitizeInput($data) {
        if (is_array($data)) {
            return array_map([self::class, 'sanitizeInput'], $data);
        }
        
        $data = trim($data);
        $data = stripslashes($data);
        $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
        return $data;
    }

    public static function sanitizeString($string) {
        return filter_var($string, FILTER_SANITIZE_STRING, FILTER_FLAG_NO_ENCODE_QUOTES);
    }

    public static function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL);
    }

    public static function validatePhone($phone) {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        return preg_match('/^[0-9]{10,11}$/', $phone);
    }

    public static function validatePassword($password) {
        return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d@$!%*?&]{8,}$/', $password);
    }

    public static function hashPassword($password) {
        if (defined('PASSWORD_ARGON2ID')) {
            return password_hash($password, PASSWORD_ARGON2ID, [
                'memory_cost' => 65536,
                'time_cost' => 4,
                'threads' => 3
            ]);
        } else {
            return password_hash($password, PASSWORD_DEFAULT, ['cost' => 12]);
        }
    }

    public static function verifyPassword($password, $hash) {
        return password_verify($password, $hash);
    }

    public static function isLoggedIn() {
        return isset($_SESSION['user_id']) && isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] === true;
    }

    public static function loginUser($user_id, $username, $rol_id = null) {
        session_regenerate_id(true);
        $_SESSION['user_id'] = $user_id;
        $_SESSION['username'] = $username;
        $_SESSION['user_logged_in'] = true;
        $_SESSION['login_time'] = time();
        $_SESSION['last_activity'] = time();
        $_SESSION['user_role'] = $rol_id;
    }

    public static function logoutUser() {
        session_unset();
        session_destroy();
        session_start();
        session_regenerate_id(true);
    }

    public static function checkSessionTimeout() {
        $timeout = 1800; // 30 dakika
        
        if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout) {
            self::logoutUser();
            return false;
        }
        
        $_SESSION['last_activity'] = time();
        return true;
    }

    public static function checkLoginAttempts($ip) {
        $max_attempts = 5;
        $lockout_time = 900; // 15 dakika
        
        if (!isset($_SESSION['login_attempts'])) {
            $_SESSION['login_attempts'] = [];
        }
        
        if (!isset($_SESSION['login_attempts'][$ip])) {
            $_SESSION['login_attempts'][$ip] = [
                'count' => 0,
                'last_attempt' => 0
            ];
        }
        
        $attempts = $_SESSION['login_attempts'][$ip];
        
        if (time() - $attempts['last_attempt'] > $lockout_time) {
            $_SESSION['login_attempts'][$ip] = [
                'count' => 0,
                'last_attempt' => 0
            ];
            return true;
        }
        
        if ($attempts['count'] >= $max_attempts) {
            return false;
        }
        
        return true;
    }

    public static function recordFailedLogin($ip) {
        if (!isset($_SESSION['login_attempts'][$ip])) {
            $_SESSION['login_attempts'][$ip] = [
                'count' => 0,
                'last_attempt' => 0
            ];
        }
        
        $_SESSION['login_attempts'][$ip]['count']++;
        $_SESSION['login_attempts'][$ip]['last_attempt'] = time();
    }

    public static function resetLoginAttempts($ip) {
        if (isset($_SESSION['login_attempts'][$ip])) {
            unset($_SESSION['login_attempts'][$ip]);
        }
    }

    public static function escapeOutput($data) {
        return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    }

    public static function validateFileUpload($file, $allowed_types = ['jpg', 'jpeg', 'png', 'gif']) {
        if (!isset($file['tmp_name']) || empty($file['tmp_name'])) {
            return false;
        }
        
        $file_info = pathinfo($file['name']);
        $extension = strtolower($file_info['extension']);
        
        if (!in_array($extension, $allowed_types)) {
            return false;
        }
        
        $mime_type = mime_content_type($file['tmp_name']);
        $allowed_mimes = [
            'image/jpeg',
            'image/jpg', 
            'image/png',
            'image/gif'
        ];
        
        return in_array($mime_type, $allowed_mimes);
    }

    public static function getClientIP() {
        $ip_keys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }

    public static function getUserRole() {
        return $_SESSION['user_role'] ?? null;
    }

    public static function hasPageAccess($page_name, $permission_type = 'okuma') {
        global $database;
        
        if (!self::isLoggedIn()) {
            return false;
        }
        
        $user_role = self::getUserRole();
        if (!$user_role) {
            return false;
        }
        
        try {
            $sql = "SELECT {$permission_type} FROM sayfa_yetkileri WHERE rol_id = ? AND sayfa_adi = ?";
            $result = $database->fetch($sql, [$user_role, $page_name]);
            
            return $result && $result[$permission_type] == 1;
        } catch (Exception $e) {
            error_log("Yetki kontrolü hatası: " . $e->getMessage());
            return false;
        }
    }

    public static function isAdmin() {
        return self::getUserRole() == 1; // Admin rol_id = 1
    }

    public static function isManager() {
        $role = self::getUserRole();
        return $role == 1 || $role == 2; // Admin veya Manager
    }
}

function requireLogin() {
    if (!Security::isLoggedIn() || !Security::checkSessionTimeout()) {
        header('Location: ../pages/login.php');
        exit();
    }
}

function validateCSRF() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (!isset($_POST['csrf_token']) || !Security::validateCSRFToken($_POST['csrf_token'])) {
            die('Güvenlik hatası: Geçersiz CSRF token.');
        }
    }
}
?>