<?php
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../config/database.php';

requireLogin();

Authorization::requirePageAccess('customers', 'okuma');

$error_message = '';
$success_message = '';
$action = $_GET['action'] ?? 'list';
$customer_id = $_GET['id'] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        validateCSRF();
        
        Authorization::requirePageAccess('customers', 'yazma');
        
        $firma_adi = Security::sanitizeInput($_POST['firma_adi'] ?? '');
        $cari_adi = Security::sanitizeInput($_POST['cari_adi'] ?? '');
        $telefon = Security::sanitizeInput($_POST['telefon'] ?? '');
        $email = Security::sanitizeInput($_POST['email'] ?? '');
        $adres = Security::sanitizeInput($_POST['adres'] ?? '');
        $vergi_no = Security::sanitizeInput($_POST['vergi_no'] ?? '');
        $cari_tipi = Security::sanitizeInput($_POST['cari_tipi'] ?? 'musteri');
        
        if (empty($firma_adi)) {
            throw new Exception('Firma adı gereklidir.');
        }
        if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Geçerli bir e-posta adresi giriniz.');
        }
        if (!empty($telefon) && !preg_match('/^[0-9\s\-\+\(\)]+$/', $telefon)) {
            throw new Exception('Geçerli bir telefon numarası giriniz.');
        }
        
        if ($action === 'add') {
            if (!empty($email)) {
                $sql = "SELECT COUNT(*) as count FROM cariler WHERE email = ?";
                $result = $database->fetch($sql, [$email]);
                if ($result['count'] > 0) {
                    throw new Exception('Bu e-posta adresi zaten kullanılıyor.');
                }
            }
            
            $sql = "INSERT INTO cariler (firma_adi, cari_adi, telefon, email, adres, vergi_no, cari_tipi) VALUES (?, ?, ?, ?, ?, ?, ?)";
            $database->execute($sql, [$firma_adi, $cari_adi, $telefon, $email, $adres, $vergi_no, $cari_tipi]);
            $success_message = 'Cari başarıyla eklendi.';
            
        } elseif ($action === 'edit' && $customer_id) {
            if (!empty($email)) {
                $sql = "SELECT COUNT(*) as count FROM cariler WHERE email = ? AND cari_id != ?";
                $result = $database->fetch($sql, [$email, $customer_id]);
                if ($result['count'] > 0) {
                    throw new Exception('Bu e-posta adresi zaten kullanılıyor.');
                }
            }
            
            $sql = "UPDATE cariler SET firma_adi = ?, cari_adi = ?, telefon = ?, email = ?, adres = ?, vergi_no = ?, cari_tipi = ? WHERE cari_id = ?";
            $database->execute($sql, [$firma_adi, $cari_adi, $telefon, $email, $adres, $vergi_no, $cari_tipi, $customer_id]);
            $success_message = 'Cari başarıyla güncellendi.';
        }
        
        if ($success_message) {
            header('Location: customers.php?message=success');
            exit();
        }
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
    }
}

if ($action === 'delete' && $customer_id && $_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        validateCSRF();
        
        Authorization::requirePageAccess('customers', 'silme');
        
        $sql = "SELECT COUNT(*) as count FROM satislar WHERE cari_id = ?";
        $result = $database->fetch($sql, [$customer_id]);
        
        if ($result['count'] > 0) {
            $error_message = 'Bu carinin satış kaydı bulunduğu için silinemez.';
        } else {
            $sql = "DELETE FROM cariler WHERE cari_id = ?";
            $database->execute($sql, [$customer_id]);
            header('Location: customers.php?message=deleted');
            exit();
        }
    } catch (Exception $e) {
        $error_message = 'Cari silinirken hata oluştu: ' . $e->getMessage();
    }
}

if (isset($_GET['message'])) {
    switch ($_GET['message']) {
        case 'success':
            $success_message = 'İşlem başarıyla tamamlandı.';
            break;
        case 'deleted':
            $success_message = 'Cari başarıyla silindi.';
            break;
    }
}

$search = Security::sanitizeInput($_GET['search'] ?? '');
$filter_type = $_GET['filter_type'] ?? '';

$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(cari_adi LIKE ? OR telefon LIKE ? OR email LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($filter_type)) {
    $where_conditions[] = "cari_tipi = ?";
    $params[] = $filter_type;
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

$sql = "SELECT c.*, 
               c.cari_bakiye as bakiye,
               COALESCE(SUM(CASE WHEN ch.hareket_tipi = 'borc' THEN ch.tutar ELSE 0 END), 0) as toplam_borc,
               COALESCE(SUM(CASE WHEN ch.hareket_tipi = 'alacak' THEN ch.tutar ELSE 0 END), 0) as toplam_odenen,
               COALESCE(SUM(ch.tutar), 0) as toplam_islem_hacmi
        FROM cariler c
        LEFT JOIN cari_hareketler ch ON c.cari_id = ch.cari_id
        $where_clause
        GROUP BY c.cari_id, c.cari_bakiye
        ORDER BY c.cari_adi ASC";
$customers = $database->fetchAll($sql, $params);

$edit_customer = null;
if ($action === 'edit' && $customer_id) {
    $sql = "SELECT * FROM cariler WHERE cari_id = ?";
    $edit_customer = $database->fetch($sql, [$customer_id]);
    if (!$edit_customer) {
        $error_message = 'Cari bulunamadı.';
        $action = 'list';
    }
}
?>
<?php include '../includes/header.php'; ?>
    <style>
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        }
        .btn {
            border-radius: 10px;
        }
        .form-control {
            border-radius: 10px;
            border: 2px solid #e9ecef;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .table th {
            background: #f8f9fa;
            border: none;
            font-weight: 600;
        }
        .badge {
            font-size: 0.75rem;
        }
    </style>
</head>
<body>
    <?php include '../includes/sidebar.php'; ?>

    
    <div class="main-content" id="mainContent">
        
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h2>
                <i class="fas fa-users me-2"></i>
                Cari Tanımı
                <?php if ($action === 'add'): ?>
                    - Yeni Cari Ekle
                <?php elseif ($action === 'edit'): ?>
                    - Cari Düzenle
                <?php endif; ?>
            </h2>
            <a href="dashboard.php" class="btn btn-outline-secondary">
                <i class="fas fa-arrow-left me-1"></i>
                Dashboard
            </a>
        </div>

        
        <div class="container-fluid">
            <?php if ($error_message): ?>
                <div class="alert alert-danger" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <?php echo Security::escapeOutput($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($success_message): ?>
                <div class="alert alert-success" role="alert">
                    <i class="fas fa-check-circle me-2"></i>
                    <?php echo Security::escapeOutput($success_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($action === 'list'): ?>
                
                <div class="row mb-4">
                    <div class="col-md-6">
                        <?php if (Authorization::getUserPagePermissions('customers')['yazma']): ?>
                        <a href="?action=add" class="btn btn-primary">
                            <i class="fas fa-plus me-2"></i>
                            Yeni Cari Ekle
                        </a>
                        <?php endif; ?>
                    </div>
                    <div class="col-md-6">
                        <form method="GET" class="d-flex">
                            <input type="text" 
                                   class="form-control me-2" 
                                   name="search" 
                                   placeholder="Cari adı, telefon veya e-posta ara..."
                                   value="<?php echo Security::escapeOutput($search); ?>">
                            <select name="filter_type" class="form-select me-2" style="width: auto;">
                                <option value="">Tüm Tipler</option>
                                <option value="musteri" <?php echo $filter_type === 'musteri' ? 'selected' : ''; ?>>Müşteri</option>
                                <option value="tedarikci" <?php echo $filter_type === 'tedarikci' ? 'selected' : ''; ?>>Tedarikçi</option>
                            </select>
                            <button type="submit" class="btn btn-outline-primary">
                                <i class="fas fa-search"></i>
                            </button>
                        </form>
                    </div>
                </div>

                <div class="card">
                    <div class="card-body">
                        <?php if (empty($customers)): ?>
                            <div class="text-center py-5">
                                <i class="fas fa-users fa-3x text-muted mb-3"></i>
                                <h5 class="text-muted">Cari bulunamadı</h5>
                                <p class="text-muted">
                                    <?php if (!empty($search) || !empty($filter_type)): ?>
                                        Arama kriterlerinize uygun cari bulunamadı.
                                    <?php else: ?>
                                        Henüz hiç cari eklenmemiş.
                                    <?php endif; ?>
                                </p>
                                <a href="?action=add" class="btn btn-primary">
                                    <i class="fas fa-plus me-2"></i>
                                    İlk Cariyi Ekle
                                </a>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Cari Adı</th>
                                            <th>Tip</th>
                                            <th>Telefon</th>
                                            <th>E-posta</th>
                                            <th>İşlem Hacmi</th>
                                            <th>İşlemler</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($customers as $customer): ?>
                                            <tr>
                                                <td>
                                                    <strong><?php echo Security::escapeOutput($customer['cari_adi']); ?></strong>
                                                    <?php if ($customer['vergi_no']): ?>
                                                        <br><small class="text-muted">VN: <?php echo Security::escapeOutput($customer['vergi_no']); ?></small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($customer['cari_tipi'] === 'musteri'): ?>
                                                        <span class="badge bg-primary">Müşteri</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-success">Tedarikçi</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($customer['telefon']): ?>
                                                        <a href="tel:<?php echo Security::escapeOutput($customer['telefon']); ?>" class="text-decoration-none">
                                                            <?php echo Security::escapeOutput($customer['telefon']); ?>
                                                        </a>
                                                    <?php else: ?>
                                                        <span class="text-muted">-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($customer['email']): ?>
                                                        <a href="mailto:<?php echo Security::escapeOutput($customer['email']); ?>" class="text-decoration-none">
                                                            <?php echo Security::escapeOutput($customer['email']); ?>
                                                        </a>
                                                    <?php else: ?>
                                                        <span class="text-muted">-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="fw-bold">₺<?php echo number_format($customer['toplam_islem_hacmi'], 2); ?></span>
                                                    <small class="text-muted d-block">
                                                        <?php if ($customer['cari_tipi'] === 'musteri'): ?>
                                                            <?php if ($customer['bakiye'] > 0): ?>
                                                                <span class="text-success">Bize Borçlu: ₺<?php echo number_format($customer['bakiye'], 2); ?></span>
                                                            <?php elseif ($customer['bakiye'] < 0): ?>
                                                                <span class="text-danger">Hata: Müşteri alacaklı olamaz!</span>
                                                            <?php else: ?>
                                                                <span class="text-muted">Bakiye: ₺0.00</span>
                                                            <?php endif; ?>
                                                        <?php else: // tedarikci ?>
                                                            <?php if ($customer['bakiye'] > 0): ?>
                                                                <span class="text-danger">Bize Borçlu: ₺<?php echo number_format($customer['bakiye'], 2); ?></span>
                                                            <?php elseif ($customer['bakiye'] < 0): ?>
                                                                <span class="text-warning">Hata: Tedarikçi alacaklı olamaz!</span>
                                                            <?php else: ?>
                                                                <span class="text-muted">Bakiye: ₺0.00</span>
                                                            <?php endif; ?>
                                                        <?php endif; ?>
                                                    </small>
                                                </td>
                                                <td>
                                                    <div class="btn-group btn-group-sm">
                                                        <a href="customer_detail.php?id=<?php echo $customer['cari_id']; ?>" 
                                                           class="btn btn-outline-info" 
                                                           title="Detay">
                                                            <i class="fas fa-eye"></i>
                                                        </a>
                                                        <?php if (Authorization::getUserPagePermissions('customers')['yazma']): ?>
                                                        <a href="?action=edit&id=<?php echo $customer['cari_id']; ?>" 
                                                           class="btn btn-outline-primary" 
                                                           title="Düzenle">
                                                            <i class="fas fa-edit"></i>
                                                        </a>
                                                        <?php endif; ?>
                                                        <?php if (Authorization::getUserPagePermissions('customers')['silme']): ?>
                                                        <button type="button" 
                                                                class="btn btn-outline-danger"
                                                                title="Sil"
                                                                onclick="confirmDelete(<?php echo $customer['cari_id']; ?>, '<?php echo Security::escapeOutput($customer['cari_adi']); ?>')">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

            <?php elseif ($action === 'add' || $action === 'edit'): ?>
                
                <div class="row justify-content-center">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <i class="fas fa-<?php echo $action === 'add' ? 'plus' : 'edit'; ?> me-2"></i>
                                    <?php echo $action === 'add' ? 'Yeni Cari Ekle' : 'Cari Düzenle'; ?>
                                </h5>
                            </div>
                            <div class="card-body">
                                <form method="POST" novalidate>
                                    <?php echo Security::getCSRFTokenInput(); ?>
                                    
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="firma_adi" class="form-label">Firma Adı *</label>
                                            <input type="text" 
                                                   class="form-control" 
                                                   id="firma_adi" 
                                                   name="firma_adi" 
                                                   value="<?php echo Security::escapeOutput($edit_customer['firma_adi'] ?? ''); ?>"
                                                   required>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="cari_adi" class="form-label">Cari Adı</label>
                                            <input type="text" 
                                                   class="form-control" 
                                                   id="cari_adi" 
                                                   name="cari_adi" 
                                                   value="<?php echo Security::escapeOutput($edit_customer['cari_adi'] ?? ''); ?>">
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-8 mb-3">
                                            <label for="cari_tipi" class="form-label">Cari Tipi *</label>
                                            <select class="form-select" id="cari_tipi" name="cari_tipi" required>
                                                <option value="musteri" <?php echo ($edit_customer['cari_tipi'] ?? 'musteri') === 'musteri' ? 'selected' : ''; ?>>Müşteri</option>
                                                <option value="tedarikci" <?php echo ($edit_customer['cari_tipi'] ?? '') === 'tedarikci' ? 'selected' : ''; ?>>Tedarikçi</option>
                                            </select>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="telefon" class="form-label">Telefon</label>
                                            <input type="tel" 
                                                   class="form-control" 
                                                   id="telefon" 
                                                   name="telefon" 
                                                   value="<?php echo Security::escapeOutput($edit_customer['telefon'] ?? ''); ?>">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="email" class="form-label">E-posta</label>
                                            <input type="email" 
                                                   class="form-control" 
                                                   id="email" 
                                                   name="email" 
                                                   value="<?php echo Security::escapeOutput($edit_customer['email'] ?? ''); ?>">
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="vergi_no" class="form-label">Vergi No</label>
                                            <input type="text" 
                                                   class="form-control" 
                                                   id="vergi_no" 
                                                   name="vergi_no" 
                                                   value="<?php echo Security::escapeOutput($edit_customer['vergi_no'] ?? ''); ?>">
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <label for="adres" class="form-label">Adres</label>
                                        <textarea class="form-control" 
                                                  id="adres" 
                                                  name="adres" 
                                                  rows="3"><?php echo Security::escapeOutput($edit_customer['adres'] ?? ''); ?></textarea>
                                    </div>

                                    <div class="d-flex justify-content-between">
                                        <a href="customers.php" class="btn btn-secondary">
                                            <i class="fas fa-arrow-left me-2"></i>
                                            Geri Dön
                                        </a>
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-save me-2"></i>
                                            <?php echo $action === 'add' ? 'Cari Ekle' : 'Güncelle'; ?>
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function confirmDelete(id, name) {
            if (confirm(`"${name}" adlı cariyi silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz.`)) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = `?action=delete&id=${id}`;
                
                const csrfInput = document.createElement('input');
                csrfInput.type = 'hidden';
                csrfInput.name = 'csrf_token';
                csrfInput.value = '<?php echo Security::generateCSRFToken(); ?>';
                form.appendChild(csrfInput);
                
                document.body.appendChild(form);
                form.submit();
            }
        }

        document.querySelector('form')?.addEventListener('submit', function(e) {
            const firmaAdi = document.getElementById('firma_adi').value.trim();
            const email = document.getElementById('email').value.trim();
            const telefon = document.getElementById('telefon').value.trim();

            if (!firmaAdi) {
                e.preventDefault();
                alert('Firma adı gereklidir.');
                return false;
            }

            if (email && !email.includes('@')) {
                e.preventDefault();
                alert('Geçerli bir e-posta adresi giriniz.');
                return false;
            }

            if (telefon && !/^[0-9\s\-\+\(\)]+$/.test(telefon)) {
                e.preventDefault();
                alert('Geçerli bir telefon numarası giriniz.');
                return false;
            }
        });
    </script>
</body>
</html>

