<?php
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../includes/error_handler.php';
require_once '../includes/stock_manager.php';

session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}
Security::checkSessionTimeout();

Authorization::requirePageAccess('stock_movements');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    Security::validateCSRFToken($_POST['csrf_token'] ?? '');
}

$stockManager = new StockManager();
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    Authorization::requirePageAccess('stock_movements', 'yazma');
    
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'stock_entry':
            try {
                $urun_id = Security::sanitizeString($_POST['urun_id']);
                $miktar = (int)$_POST['miktar'];
                $birim_fiyat = !empty($_POST['birim_fiyat']) ? (float)$_POST['birim_fiyat'] : null;
                $aciklama = Security::sanitizeString($_POST['aciklama'] ?? '');
                $referans_no = Security::sanitizeString($_POST['referans_no'] ?? '');
                $tedarikci_id = !empty($_POST['tedarikci_id']) ? Security::sanitizeString($_POST['tedarikci_id']) : null;
                $odenen_tutar = !empty($_POST['odenen_tutar']) ? (float)$_POST['odenen_tutar'] : 0;
                
                if ($tedarikci_id && $birim_fiyat > 0) {
                    // Tedarikçili stok alımı - Merkezi BalanceManager kullan
                    require_once '../includes/balance_manager.php';
                    $database = Database::getInstance();
                    $balanceManager = new BalanceManager($database);
                    
                    $urunler = [
                        [
                            'urun_id' => $urun_id,
                            'miktar' => $miktar,
                            'alis_fiyati' => $birim_fiyat
                        ]
                    ];
                    
                    $result = $balanceManager->processStockPurchase(
                        $tedarikci_id, 
                        $urunler, 
                        $odenen_tutar, 
                        $aciklama . ($referans_no ? " - Ref: $referans_no" : "")
                    );
                } else {
                    // Basit stok girişi - StockManager kullan
                    $result = $stockManager->stockEntry($urun_id, $miktar, $birim_fiyat, $aciklama, $referans_no);
                }
                
                $message = $result['message'];
                $messageType = $result['success'] ? 'success' : 'danger';
                
            } catch (Exception $e) {
                ErrorHandler::logError('stock_entry_with_supplier', $e->getMessage());
                $message = 'Stok girişi sırasında hata oluştu: ' . $e->getMessage();
                $messageType = 'danger';
            }
            break;
            
        case 'stock_exit':
            $result = $stockManager->stockExit(
                Security::sanitizeString($_POST['urun_id']),
                (int)$_POST['miktar'],
                null, // Birim fiyat sistem tarafından otomatik alınacak
                Security::sanitizeString($_POST['aciklama'] ?? ''),
                Security::sanitizeString($_POST['referans_no'] ?? '')
            );
            $message = $result['message'];
            $messageType = $result['success'] ? 'success' : 'danger';
            break;
            
        case 'start_count':
            $result = $stockManager->startStockCount(
                Security::sanitizeString($_POST['sayim_adi']),
                Security::sanitizeString($_POST['aciklama'] ?? '')
            );
            $message = $result['message'];
            $messageType = $result['success'] ? 'success' : 'danger';
            break;
            
        case 'update_levels':
            $result = $stockManager->updateStockLevels(
                Security::sanitizeString($_POST['urun_id']),
                (int)$_POST['minimum_stok'],
                !empty($_POST['maksimum_stok']) ? (int)$_POST['maksimum_stok'] : null,
                !empty($_POST['kritik_seviye']) ? (int)$_POST['kritik_seviye'] : null
            );
            $message = $result['message'];
            $messageType = $result['success'] ? 'success' : 'danger';
            break;
    }
}

$products = [];
$suppliers = [];
try {
    $database = Database::getInstance();
    $stmt = $database->getConnection()->query("SELECT urun_id, urun_adi, barkod, stok_miktari, alis_fiyati, satis_fiyati FROM urunler ORDER BY urun_adi");
    $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Tedarikçileri çek
    $stmt = $database->getConnection()->query("SELECT cari_id, firma_adi, cari_adi FROM cariler WHERE cari_tipi = 'tedarikci' ORDER BY firma_adi");
    $suppliers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    ErrorHandler::logError('stock_movements_page', $e->getMessage());
}

$stockStatus = $stockManager->getStockStatus();
$criticalAlerts = $stockManager->getCriticalStockAlerts();
$recentMovements = $stockManager->getStockHistory(null, 20);

$userInfo = Authorization::getNavbarUserInfo();

$page_title = "Stok Hareketleri";
?>
<?php include '../includes/header.php'; ?>
    <style>
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            transition: transform 0.3s ease;
        }
        .card:hover {
            transform: translateY(-5px);
        }
        .alert-critical {
            background-color: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }
        .alert-low {
            background-color: #fff3cd;
            border-color: #ffeaa7;
            color: #856404;
        }
        .stock-status-kritik {
            background-color: #dc3545;
            color: white;
        }
        .stock-status-dusuk {
            background-color: #ffc107;
            color: #212529;
        }
        .stock-status-normal {
            background-color: #28a745;
            color: white;
        }
        .stock-status-yuksek {
            background-color: #17a2b8;
            color: white;
        }
    </style>

<?php include '../includes/sidebar.php'; ?>

<main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
    <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
        <h1 class="h2">
            <button class="btn btn-outline-secondary d-md-none me-2" type="button" id="sidebarToggle">
                <i class="fas fa-bars"></i>
            </button>
            <?php echo $page_title; ?>
        </h1>
        <div class="btn-toolbar mb-2 mb-md-0">
            <a href="dashboard.php" class="btn btn-sm btn-outline-secondary">
                <i class="fas fa-arrow-left me-1"></i>Dashboard'a Dön
            </a>
        </div>
    </div>

                <?php if ($message): ?>
                <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                    <?= htmlspecialchars($message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                
                <?php if (!empty($criticalAlerts)): ?>
                <div class="alert alert-warning" role="alert">
                    <h6><i class="fas fa-exclamation-triangle me-2"></i>Kritik Stok Uyarıları</h6>
                    <div class="row">
                        <?php foreach ($criticalAlerts as $alert): ?>
                        <div class="col-md-6 mb-2">
                            <span class="badge bg-<?= $alert['stok_durumu'] == 'Kritik' ? 'danger' : 'warning' ?>">
                                <?= htmlspecialchars($alert['urun_adi']) ?> - <?= $alert['stok_miktari'] ?> adet
                            </span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>

                
                <ul class="nav nav-tabs" id="stockTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="entry-tab" data-bs-toggle="tab" data-bs-target="#entry" type="button" role="tab">
                            <i class="fas fa-plus me-2"></i>Stok Giriş
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="exit-tab" data-bs-toggle="tab" data-bs-target="#exit" type="button" role="tab">
                            <i class="fas fa-minus me-2"></i>Stok Çıkış
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="count-tab" data-bs-toggle="tab" data-bs-target="#count" type="button" role="tab">
                            <i class="fas fa-clipboard-list me-2"></i>Stok Sayım
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="status-tab" data-bs-toggle="tab" data-bs-target="#status" type="button" role="tab">
                            <i class="fas fa-chart-pie me-2"></i>Stok Durumu
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="history-tab" data-bs-toggle="tab" data-bs-target="#history" type="button" role="tab">
                            <i class="fas fa-history me-2"></i>Hareket Geçmişi
                        </button>
                    </li>
                </ul>

                <div class="tab-content" id="stockTabsContent">
                    
                    <div class="tab-pane fade show active" id="entry" role="tabpanel">
                        <div class="card mt-3">
                            <div class="card-header">
                                <h5><i class="fas fa-plus me-2"></i>Stok Giriş İşlemi</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST">
                                    <input type="hidden" name="csrf_token" value="<?= Security::generateCSRFToken() ?>">
                                    <input type="hidden" name="action" value="stock_entry">
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="entry_urun_id" class="form-label">Ürün *</label>
                                                <select class="form-select" id="entry_urun_id" name="urun_id" required>
                                                    <option value="">Ürün Seçin</option>
                                                    <?php foreach ($products as $product): ?>
                                                    <option value="<?= $product['urun_id'] ?>">
                                                        <?= htmlspecialchars($product['urun_adi']) ?> 
                                                        (Mevcut: <?= $product['stok_miktari'] ?>)
                                                    </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="entry_miktar" class="form-label">Miktar *</label>
                                                <input type="number" class="form-control" id="entry_miktar" name="miktar" min="1" required onchange="calculateEntryTotal()">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-12">
                                            <div class="mb-3">
                                                <label for="entry_tedarikci_id" class="form-label">Tedarikçi</label>
                                                <select class="form-select" id="entry_tedarikci_id" name="tedarikci_id">
                                                    <option value="">Tedarikçi Seçin (İsteğe Bağlı)</option>
                                                    <?php foreach ($suppliers as $supplier): ?>
                                                    <option value="<?= $supplier['cari_id'] ?>">
                                                        <?= htmlspecialchars($supplier['firma_adi'] ?: $supplier['cari_adi']) ?>
                                                    </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="mb-3">
                                                <label for="entry_birim_fiyat" class="form-label">Birim Fiyat *</label>
                                                <input type="number" step="0.01" class="form-control" id="entry_birim_fiyat" name="birim_fiyat" required onchange="calculateEntryTotal()">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="mb-3">
                                                <label for="entry_toplam_tutar" class="form-label">Toplam Tutar</label>
                                                <input type="text" class="form-control" id="entry_toplam_tutar" readonly style="background-color: #f8f9fa;">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="mb-3">
                                                <label for="entry_referans_no" class="form-label">Referans No</label>
                                                <input type="text" class="form-control" id="entry_referans_no" name="referans_no">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row" id="payment_section" style="display: none;">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="entry_odenen_tutar" class="form-label">Ödenen Tutar</label>
                                                <input type="number" step="0.01" class="form-control" id="entry_odenen_tutar" name="odenen_tutar" min="0" onchange="calculateRemainingAmount()">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="entry_kalan_tutar" class="form-label">Kalan Tutar</label>
                                                <input type="text" class="form-control" id="entry_kalan_tutar" readonly style="background-color: #f8f9fa;">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="entry_aciklama" class="form-label">Açıklama</label>
                                        <textarea class="form-control" id="entry_aciklama" name="aciklama" rows="3"></textarea>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-success">
                                        <i class="fas fa-plus me-2"></i>Stok Giriş Yap
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>

                    
                    <div class="tab-pane fade" id="exit" role="tabpanel">
                        <div class="card mt-3">
                            <div class="card-header">
                                <h5><i class="fas fa-minus me-2"></i>Stok Çıkış İşlemi</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST">
                                    <input type="hidden" name="csrf_token" value="<?= Security::generateCSRFToken() ?>">
                                    <input type="hidden" name="action" value="stock_exit">
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="exit_urun_id" class="form-label">Ürün *</label>
                                                <select class="form-select" id="exit_urun_id" name="urun_id" required>
                                                    <option value="">Ürün Seçin</option>
                                                    <?php foreach ($products as $product): ?>
                                                    <option value="<?= $product['urun_id'] ?>">
                                                        <?= htmlspecialchars($product['urun_adi']) ?> 
                                                        (Mevcut: <?= $product['stok_miktari'] ?>)
                                                    </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="exit_miktar" class="form-label">Miktar *</label>
                                                <input type="number" class="form-control" id="exit_miktar" name="miktar" min="1" required onchange="calculateExitTotal()">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="mb-3">
                                                <label for="exit_birim_fiyat" class="form-label">Satış Fiyatı (Sistem)</label>
                                                <input type="text" class="form-control" id="exit_birim_fiyat" readonly style="background-color: #f8f9fa;" placeholder="Ürün seçin">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="mb-3">
                                                <label for="exit_toplam_tutar" class="form-label">Toplam Tutar</label>
                                                <input type="text" class="form-control" id="exit_toplam_tutar" readonly style="background-color: #f8f9fa;">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="mb-3">
                                                <label for="exit_referans_no" class="form-label">Referans No</label>
                                                <input type="text" class="form-control" id="exit_referans_no" name="referans_no">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="exit_aciklama" class="form-label">Açıklama</label>
                                        <textarea class="form-control" id="exit_aciklama" name="aciklama" rows="3"></textarea>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-danger">
                                        <i class="fas fa-minus me-2"></i>Stok Çıkış Yap
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>

                    
                    <div class="tab-pane fade" id="count" role="tabpanel">
                        <div class="card mt-3">
                            <div class="card-header">
                                <h5><i class="fas fa-clipboard-list me-2"></i>Stok Sayım</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST">
                                    <input type="hidden" name="csrf_token" value="<?= Security::generateCSRFToken() ?>">
                                    <input type="hidden" name="action" value="start_count">
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="sayim_adi" class="form-label">Sayım Adı *</label>
                                                <input type="text" class="form-control" id="sayim_adi" name="sayim_adi" required>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="count_aciklama" class="form-label">Açıklama</label>
                                                <textarea class="form-control" id="count_aciklama" name="aciklama" rows="3"></textarea>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-clipboard-list me-2"></i>Sayım Başlat
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>

                    
                    <div class="tab-pane fade" id="status" role="tabpanel">
                        <div class="card mt-3">
                            <div class="card-header">
                                <h5><i class="fas fa-chart-pie me-2"></i>Stok Durumu</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Ürün Adı</th>
                                                <th>Barkod</th>
                                                <th>Mevcut Stok</th>
                                                <th>Min. Stok</th>
                                                <th>Durum</th>
                                                <th>Toplam Değer</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($stockStatus as $item): ?>
                                            <tr>
                                                <td><?= htmlspecialchars($item['urun_adi']) ?></td>
                                                <td><?= htmlspecialchars($item['barkod']) ?></td>
                                                <td><?= $item['stok_miktari'] ?></td>
                                                <td><?= $item['minimum_stok'] ?></td>
                                                <td>
                                                    <span class="badge stock-status-<?= strtolower($item['stok_durumu']) ?>">
                                                        <?= $item['stok_durumu'] ?>
                                                    </span>
                                                </td>
                                                <td><?= number_format($item['toplam_alis_degeri'], 2) ?> ₺</td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>

                    
                    <div class="tab-pane fade" id="history" role="tabpanel">
                        <div class="card mt-3">
                            <div class="card-header">
                                <h5><i class="fas fa-history me-2"></i>Son Hareketler</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Tarih</th>
                                                <th>Ürün</th>
                                                <th>Hareket</th>
                                                <th>Miktar</th>
                                                <th>Önceki Stok</th>
                                                <th>Yeni Stok</th>
                                                <th>Kullanıcı</th>
                                                <th>Açıklama</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($recentMovements as $movement): ?>
                                            <tr>
                                                <td><?= date('d.m.Y H:i', strtotime($movement['tarih'])) ?></td>
                                                <td><?= htmlspecialchars($movement['urun_adi']) ?></td>
                                                <td>
                                                    <span class="badge bg-<?= $movement['hareket_tipi'] == 'giris' ? 'success' : ($movement['hareket_tipi'] == 'cikis' ? 'danger' : 'info') ?>">
                                                        <?= ucfirst($movement['hareket_tipi']) ?>
                                                    </span>
                                                </td>
                                                <td><?= $movement['miktar'] ?></td>
                                                <td><?= $movement['onceki_stok'] ?></td>
                                                <td><?= $movement['yeni_stok'] ?></td>
                                                <td><?= htmlspecialchars($movement['kullanici_adi']) ?></td>
                                                <td><?= htmlspecialchars($movement['aciklama']) ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
</main>

<script>
// Ürün verileri (PHP'den JavaScript'e aktarım)
const products = <?= json_encode($products) ?>;

function updateProductPrices(selectElement, type) {
    const selectedProductId = selectElement.value;
    const selectedProduct = products.find(p => p.urun_id == selectedProductId);
    
    if (selectedProduct) {
        if (type === 'entry') {
            document.getElementById('entry_birim_fiyat').value = parseFloat(selectedProduct.alis_fiyati).toFixed(2);
            calculateEntryTotal();
        } else if (type === 'exit') {
            document.getElementById('exit_birim_fiyat').value = '₺' + parseFloat(selectedProduct.satis_fiyati).toFixed(2);
            calculateExitTotal();
        }
    } else {
        if (type === 'entry') {
            document.getElementById('entry_birim_fiyat').value = '';
            document.getElementById('entry_toplam_tutar').value = '';
        } else if (type === 'exit') {
            document.getElementById('exit_birim_fiyat').value = '';
            document.getElementById('exit_toplam_tutar').value = '';
        }
    }
}

function calculateEntryTotal() {
    const miktar = parseFloat(document.getElementById('entry_miktar').value) || 0;
    const birimFiyat = parseFloat(document.getElementById('entry_birim_fiyat').value) || 0;
    const toplamTutar = miktar * birimFiyat;
    
    document.getElementById('entry_toplam_tutar').value = toplamTutar > 0 ? '₺' + toplamTutar.toFixed(2) : '';
    calculateRemainingAmount();
}

function calculateRemainingAmount() {
    const toplamTutarText = document.getElementById('entry_toplam_tutar').value;
    const toplamTutar = parseFloat(toplamTutarText.replace('₺', '')) || 0;
    const odenenTutar = parseFloat(document.getElementById('entry_odenen_tutar').value) || 0;
    const kalanTutar = toplamTutar - odenenTutar;
    
    document.getElementById('entry_kalan_tutar').value = kalanTutar > 0 ? '₺' + kalanTutar.toFixed(2) : '₺0.00';
}

// Tedarikçi seçildiğinde ödeme bölümünü göster
document.getElementById('entry_tedarikci_id').addEventListener('change', function() {
    const paymentSection = document.getElementById('payment_section');
    if (this.value) {
        paymentSection.style.display = 'block';
    } else {
        paymentSection.style.display = 'none';
        document.getElementById('entry_odenen_tutar').value = '';
        document.getElementById('entry_kalan_tutar').value = '';
    }
});

function calculateExitTotal() {
    const miktar = parseFloat(document.getElementById('exit_miktar').value) || 0;
    const birimFiyatText = document.getElementById('exit_birim_fiyat').value;
    const birimFiyat = parseFloat(birimFiyatText.replace('₺', '')) || 0;
    const toplamTutar = miktar * birimFiyat;
    
    document.getElementById('exit_toplam_tutar').value = toplamTutar > 0 ? '₺' + toplamTutar.toFixed(2) : '';
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    const entryMiktar = document.getElementById('entry_miktar');
    const exitMiktar = document.getElementById('exit_miktar');
    const entryUrunSelect = document.getElementById('entry_urun_id');
    const exitUrunSelect = document.getElementById('exit_urun_id');
    
    if (entryMiktar) {
        entryMiktar.addEventListener('input', calculateEntryTotal);
    }
    if (exitMiktar) {
        exitMiktar.addEventListener('input', calculateExitTotal);
    }
    if (entryUrunSelect) {
        entryUrunSelect.addEventListener('change', function() {
            updateProductPrices(this, 'entry');
        });
    }
    if (exitUrunSelect) {
        exitUrunSelect.addEventListener('change', function() {
            updateProductPrices(this, 'exit');
        });
    }
});
</script>

<?php include '../includes/footer.php'; ?>