<?php

/**
 * Satış Tutarsızlıklarını Düzelt
 * kalan_tutar hesaplamasını düzeltir
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/transaction_manager.php';

// CLI modunda çalıştığını kontrol et
if (php_sapi_name() !== 'cli') {
    // Web'den çalıştırılıyorsa
    echo "<!DOCTYPE html><html><head><meta charset='UTF-8'><title>Satış Tutarsızlıklarını Düzelt</title>";
    echo "<style>body{font-family:monospace;padding:20px;background:#f5f5f5;}";
    echo ".success{color:green;}.error{color:red;}.info{color:blue;}</style></head><body>";
    echo "<h2>Satış Tutarsızlıklarını Düzelt</h2><pre>";
}

echo "==============================================\n";
echo "  SATIŞ TUTARSIZLIKLARINI DÜZELT\n";
echo "==============================================\n\n";

try {
    $database = Database::getInstance();
    $transactionManager = new TransactionManager($database);
    
    // 1. Tutarsız satışları bul
    echo "1. Tutarsız satışlar tespit ediliyor...\n";
    echo "---------------------------------------------\n";
    
    $sql = "SELECT satis_id, toplam_tutar, odenen_tutar, kalan_tutar,
                   (toplam_tutar - odenen_tutar) as calculated_kalan,
                   ABS(kalan_tutar - (toplam_tutar - odenen_tutar)) as fark
            FROM satislar 
            WHERE ABS(kalan_tutar - (toplam_tutar - odenen_tutar)) > 0.01
            ORDER BY satis_id";
    
    $inconsistentSales = $database->fetchAll($sql);
    
    if (empty($inconsistentSales)) {
        echo "✓ Tutarsız satış bulunamadı!\n\n";
        exit(0);
    }
    
    echo "✗ " . count($inconsistentSales) . " tutarsız satış bulundu:\n\n";
    
    foreach ($inconsistentSales as $sale) {
        echo "  Satış ID: {$sale['satis_id']}\n";
        echo "    Toplam Tutar: " . number_format($sale['toplam_tutar'], 2) . " TL\n";
        echo "    Ödenen Tutar: " . number_format($sale['odenen_tutar'], 2) . " TL\n";
        echo "    Kayıtlı Kalan: " . number_format($sale['kalan_tutar'], 2) . " TL\n";
        echo "    Hesaplanan Kalan: " . number_format($sale['calculated_kalan'], 2) . " TL\n";
        echo "    Fark: " . number_format($sale['fark'], 2) . " TL\n\n";
    }
    
    // 2. Kullanıcıdan onay al (CLI modunda)
    if (php_sapi_name() === 'cli') {
        echo "Bu satışları düzeltmek istiyor musunuz? (E/H): ";
        $handle = fopen("php://stdin", "r");
        $line = fgets($handle);
        fclose($handle);
        
        if (trim(strtoupper($line)) !== 'E') {
            echo "\nİşlem iptal edildi.\n";
            exit(0);
        }
    }
    
    // 3. Düzeltme işlemini başlat
    echo "\n2. Satışlar düzeltiliyor...\n";
    echo "---------------------------------------------\n";
    
    $transactionManager->begin();
    
    $fixedCount = 0;
    $errorCount = 0;
    
    foreach ($inconsistentSales as $sale) {
        try {
            $correctKalan = $sale['toplam_tutar'] - $sale['odenen_tutar'];
            
            $sql = "UPDATE satislar 
                    SET kalan_tutar = ? 
                    WHERE satis_id = ?";
            
            $database->execute($sql, [$correctKalan, $sale['satis_id']]);
            
            echo "  ✓ Satış ID {$sale['satis_id']} düzeltildi: ";
            echo number_format($sale['kalan_tutar'], 2) . " → " . number_format($correctKalan, 2) . " TL\n";
            
            $fixedCount++;
            
        } catch (Exception $e) {
            echo "  ✗ Satış ID {$sale['satis_id']} düzeltilemedi: " . $e->getMessage() . "\n";
            $errorCount++;
        }
    }
    
    $transactionManager->commit();
    
    echo "\n3. Özet\n";
    echo "---------------------------------------------\n";
    echo "  Düzeltilen: $fixedCount\n";
    echo "  Hata: $errorCount\n";
    echo "  Toplam: " . count($inconsistentSales) . "\n\n";
    
    // 4. Doğrulama
    echo "4. Doğrulama yapılıyor...\n";
    echo "---------------------------------------------\n";
    
    $sql = "SELECT COUNT(*) as count 
            FROM satislar 
            WHERE ABS(kalan_tutar - (toplam_tutar - odenen_tutar)) > 0.01";
    
    $remainingIssues = $database->fetch($sql)['count'];
    
    if ($remainingIssues == 0) {
        echo "✓ Tüm satışlar tutarlı!\n\n";
        echo "==============================================\n";
        echo "  İŞLEM BAŞARILI!\n";
        echo "==============================================\n\n";
    } else {
        echo "✗ Hala $remainingIssues tutarsız satış var!\n\n";
        echo "==============================================\n";
        echo "  İŞLEM TAMAMLANDI (UYARILAR VAR)\n";
        echo "==============================================\n\n";
    }
    
} catch (Exception $e) {
    if (isset($transactionManager) && $transactionManager->isActive()) {
        $transactionManager->rollback();
    }
    echo "\nHATA: " . $e->getMessage() . "\n\n";
    exit(1);
}

if (php_sapi_name() !== 'cli') {
    echo "</pre></body></html>";
}

?>
